<?php
// submit_feedback.php - Handle feedback submission via AJAX
require_once 'config.php';

header('Content-Type: application/json');

// Check if user is logged in
if (!isLoggedIn()) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit();
}

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode(['success' => false, 'message' => 'Invalid request method']);
    exit();
}

$call_id = intval($_POST['call_id'] ?? 0);
$notes = sanitizeInput($_POST['notes'] ?? '');
$agent_phone = getAgentPhone();

// Validate input
if ($call_id <= 0) {
    echo json_encode(['success' => false, 'message' => 'Invalid call ID']);
    exit();
}

if (empty($notes)) {
    echo json_encode(['success' => false, 'message' => 'Feedback notes are required']);
    exit();
}

try {
    $conn = getDBConnection();
    
    // Verify the call belongs to this agent
    $verify_stmt = $conn->prepare("SELECT id FROM call_logs WHERE id = ? AND agent_contact = ?");
    $verify_stmt->bind_param("is", $call_id, $agent_phone);
    $verify_stmt->execute();
    $verify_result = $verify_stmt->get_result();
    
    if ($verify_result->num_rows === 0) {
        $verify_stmt->close();
        $conn->close();
        echo json_encode(['success' => false, 'message' => 'Call not found or access denied']);
        exit();
    }
    $verify_stmt->close();
    
    // Check if feedback already exists
    $check_stmt = $conn->prepare("SELECT id FROM call_notes WHERE call_id = ?");
    $check_stmt->bind_param("i", $call_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows > 0) {
        // Update existing feedback
        $existing_note = $check_result->fetch_assoc();
        $check_stmt->close();
        
        $update_stmt = $conn->prepare("UPDATE call_notes SET notes = ?, updated_at = NOW() WHERE id = ?");
        $update_stmt->bind_param("si", $notes, $existing_note['id']);
        
        if ($update_stmt->execute()) {
            $update_stmt->close();
            $conn->close();
            echo json_encode(['success' => true, 'message' => 'Feedback updated successfully']);
        } else {
            $update_stmt->close();
            $conn->close();
            echo json_encode(['success' => false, 'message' => 'Failed to update feedback']);
        }
    } else {
        // Insert new feedback
        $check_stmt->close();
        
        $insert_stmt = $conn->prepare("INSERT INTO call_notes (call_id, agent_phone, notes, created_at) VALUES (?, ?, ?, NOW())");
        $insert_stmt->bind_param("iss", $call_id, $agent_phone, $notes);
        
        if ($insert_stmt->execute()) {
            $insert_stmt->close();
            $conn->close();
            echo json_encode(['success' => true, 'message' => 'Feedback submitted successfully']);
        } else {
            $insert_stmt->close();
            $conn->close();
            echo json_encode(['success' => false, 'message' => 'Failed to submit feedback']);
        }
    }
    
} catch (Exception $e) {
    error_log("Feedback submission error: " . $e->getMessage());
    echo json_encode(['success' => false, 'message' => 'An error occurred. Please try again.']);
}
?>