<?php
// profile.php - Agent profile page
require_once 'config.php';
requireLogin();
regenerateSession();

$agent_phone = getAgentPhone();
$agent_name = $_SESSION['agent_name'] ?? 'Agent';
$agent_level = $_SESSION['agent_level'] ?? 3;

// Map level to role name
$role_mapping = [
    1 => 'Director',
    2 => 'Office Admin',
    3 => 'Field Agent'
];
$agent_role = $role_mapping[$agent_level] ?? 'Field Agent';

$conn = getDBConnection();

// Get total calls count
$count_stmt = $conn->prepare("SELECT COUNT(*) as total FROM call_logs WHERE agent_contact = ?");
$count_stmt->bind_param("s", $agent_phone);
$count_stmt->execute();
$total_calls = $count_stmt->get_result()->fetch_assoc()['total'];
$count_stmt->close();

// Get uncommented calls count for badge
$badge_stmt = $conn->prepare("
    SELECT COUNT(*) as count 
    FROM call_logs cl 
    LEFT JOIN call_notes cn ON cl.id = cn.call_id 
    WHERE cl.agent_contact = ? AND cn.id IS NULL
");
$badge_stmt->bind_param("s", $agent_phone);
$badge_stmt->execute();
$uncommented_count = $badge_stmt->get_result()->fetch_assoc()['count'];
$badge_stmt->close();

// Get additional stats
$stats_stmt = $conn->prepare("
    SELECT 
        COUNT(*) as calls_with_feedback,
        SUM(cl.duration) as total_duration
    FROM call_logs cl 
    INNER JOIN call_notes cn ON cl.id = cn.call_id 
    WHERE cl.agent_contact = ?
");
$stats_stmt->bind_param("s", $agent_phone);
$stats_stmt->execute();
$stats = $stats_stmt->get_result()->fetch_assoc();
$stats_stmt->close();

$calls_with_feedback = $stats['calls_with_feedback'] ?? 0;
$total_duration_seconds = $stats['total_duration'] ?? 0;
$total_duration_hours = floor($total_duration_seconds / 3600);
$total_duration_mins = floor(($total_duration_seconds % 3600) / 60);

$conn->close();

// Get initials for avatar
$name_parts = explode(' ', $agent_name);
$initials = '';
if (count($name_parts) >= 2) {
    $initials = strtoupper(substr($name_parts[0], 0, 1) . substr($name_parts[1], 0, 1));
} else {
    $initials = strtoupper(substr($agent_name, 0, 2));
}

// Format role
$role_display = $agent_role;
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - Agent Portal</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="header">
        <div class="header-content">
            <h1>My Profile</h1>
            <a href="logout.php" class="logout-btn">Logout</a>
        </div>
    </div>

    <div class="container profile-container">
        <div class="profile-card">
            <div class="profile-avatar"><?php echo $initials; ?></div>
            <h2 class="profile-name"><?php echo htmlspecialchars($agent_name); ?></h2>
            <p class="profile-role"><?php echo htmlspecialchars($role_display); ?></p>

            <div class="profile-stats">
                <div class="stat-item">
                    <div class="stat-label">Total Calls Received</div>
                    <div class="stat-value"><?php echo number_format($total_calls); ?></div>
                </div>
            </div>
        </div>

        <!-- Additional Stats Card -->
        <div class="profile-card" style="margin-top: 20px;">
            <h3 style="margin-bottom: 20px; font-size: 18px; color: var(--gray-800);">Call Statistics</h3>
            
            <div class="profile-stats">
                <div class="stat-item">
                    <div class="stat-label">Calls with Feedback</div>
                    <div class="stat-value" style="font-size: 24px;"><?php echo number_format($calls_with_feedback); ?></div>
                </div>
                
                <div class="stat-item" style="margin-top: 20px;">
                    <div class="stat-label">Pending Feedback</div>
                    <div class="stat-value" style="font-size: 24px; color: var(--danger-color);">
                        <?php echo number_format($uncommented_count); ?>
                    </div>
                </div>

                <div class="stat-item" style="margin-top: 20px;">
                    <div class="stat-label">Total Talk Time</div>
                    <div class="stat-value" style="font-size: 24px;">
                        <?php echo $total_duration_hours; ?>h <?php echo $total_duration_mins; ?>m
                    </div>
                </div>
            </div>
        </div>

        <!-- Account Info Card -->
        <div class="profile-card" style="margin-top: 20px;">
            <h3 style="margin-bottom: 20px; font-size: 18px; color: var(--gray-800);">Account Information</h3>
            
            <div style="text-align: left;">
                <div style="margin-bottom: 15px;">
                    <div style="font-size: 13px; color: var(--gray-600); margin-bottom: 5px;">Phone Number</div>
                    <div style="font-size: 15px; color: var(--gray-900);"><?php echo htmlspecialchars($agent_phone); ?></div>
                </div>
                
                <div style="margin-bottom: 15px;">
                    <div style="font-size: 13px; color: var(--gray-600); margin-bottom: 5px;">Role</div>
                    <div style="font-size: 15px; color: var(--gray-900);"><?php echo htmlspecialchars($role_display); ?></div>
                </div>

                <div style="margin-bottom: 15px;">
                    <div style="font-size: 13px; color: var(--gray-600); margin-bottom: 5px;">Feedback Completion Rate</div>
                    <div style="font-size: 15px; color: var(--gray-900);">
                        <?php 
                        if ($total_calls > 0) {
                            $completion_rate = ($calls_with_feedback / $total_calls) * 100;
                            echo number_format($completion_rate, 1) . '%';
                        } else {
                            echo 'N/A';
                        }
                        ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <a href="home.php" class="nav-item">
            <div class="nav-icon">🏠</div>
            <div class="nav-label">Home</div>
        </a>
        <a href="calls.php" class="nav-item">
            <div class="nav-icon">
                📋
                <?php if ($uncommented_count > 0): ?>
                    <span class="nav-badge"><?php echo $uncommented_count; ?></span>
                <?php endif; ?>
            </div>
            <div class="nav-label">Calls</div>
        </a>
        <a href="profile.php" class="nav-item active">
            <div class="nav-icon">👤</div>
            <div class="nav-label">Profile</div>
        </a>
    </nav>

    <script>
        // Mobile viewport check
        function checkViewport() {
            if (window.innerWidth > 756) {
                document.body.innerHTML = `
                    <div class="error-404">
                        <h1>404</h1>
                        <p>This application is only accessible on mobile devices.</p>
                    </div>
                `;
            }
        }
        
        checkViewport();
        window.addEventListener('resize', checkViewport);
    </script>
</body>
</html>