<?php
// login.php - Agent login page
require_once 'config.php';

// If already logged in, redirect to home
if (isLoggedIn()) {
    header('Location: home.php');
    exit();
}

$error = '';
$success = '';

// Handle login submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $phone = sanitizeInput($_POST['phone'] ?? '');
    $password = $_POST['password'] ?? '';
    
    if (empty($phone) || empty($password)) {
        $error = 'Please provide both phone number and password.';
    } else {
        $conn = getDBConnection();
        
        // Check in agents table
        $stmt = $conn->prepare("SELECT id, password_hash, name, level, contact FROM agents WHERE contact = ? AND status != 'offline'");
        $stmt->bind_param("s", $phone);
        $stmt->execute();
        $result = $stmt->get_result();
        
        if ($result->num_rows === 1) {
            $agent = $result->fetch_assoc();
            
            // Check if password is set
            if (empty($agent['password_hash'])) {
                $error = 'Your account does not have a password set. Please contact your administrator.';
            } elseif (password_verify($password, $agent['password_hash'])) {
                // Login successful
                session_regenerate_id(true);
                $_SESSION['agent_id'] = $agent['id'];
                $_SESSION['agent_phone'] = $agent['contact'];
                $_SESSION['agent_name'] = $agent['name'];
                $_SESSION['agent_level'] = $agent['level'];
                $_SESSION['last_regeneration'] = time();
                
                // Update last login
                $update_stmt = $conn->prepare("UPDATE agents SET last_login = NOW() WHERE id = ?");
                $update_stmt->bind_param("i", $agent['id']);
                $update_stmt->execute();
                $update_stmt->close();
                
                $stmt->close();
                $conn->close();
                
                header('Location: home.php');
                exit();
            } else {
                $error = 'Invalid phone number or password.';
            }
        } else {
            $error = 'Invalid phone number or password.';
        }
        
        $stmt->close();
        $conn->close();
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Agent Login - Call Center Portal</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="login-container">
        <div class="login-box">
            <h1>Agent Portal</h1>
            <p>Sign in to access your dashboard</p>
            
            <?php if ($error): ?>
                <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
            <?php endif; ?>
            
            <form method="POST" action="login.php">
                <div class="form-group">
                    <label for="phone">Phone Number</label>
                    <input 
                        type="tel" 
                        id="phone" 
                        name="phone" 
                        placeholder="e.g., +254712345678" 
                        required 
                        autocomplete="tel"
                        value="<?php echo htmlspecialchars($_POST['phone'] ?? ''); ?>"
                    >
                </div>
                
                <div class="form-group">
                    <label for="password">Password</label>
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        placeholder="Enter your password" 
                        required
                        autocomplete="current-password"
                    >
                </div>
                
                <button type="submit" class="btn">Sign In</button>
            </form>
        </div>
    </div>

    <script>
        // Mobile viewport check
        function checkViewport() {
            if (window.innerWidth > 756) {
                document.body.innerHTML = `
                    <div class="error-404">
                        <h1>404</h1>
                        <p>This application is only accessible on mobile devices.</p>
                        <p>Please access from a device with screen width ≤ 756px</p>
                    </div>
                `;
            }
        }
        
        checkViewport();
        window.addEventListener('resize', checkViewport);
    </script>
</body>
</html>