<?php
// calls.php - Calls page with uncommented/commented sections
require_once 'config.php';
requireLogin();
regenerateSession();

$agent_phone = getAgentPhone();
$conn = getDBConnection();

// Get uncommented calls
$uncommented_sql = "
    SELECT cl.id, cl.call_date, cl.client_contact, cl.duration, cl.status 
    FROM call_logs cl 
    LEFT JOIN call_notes cn ON cl.id = cn.call_id 
    WHERE cl.agent_contact = ? AND cn.id IS NULL 
    ORDER BY cl.call_date DESC
";
$uncommented_stmt = $conn->prepare($uncommented_sql);
$uncommented_stmt->bind_param("s", $agent_phone);
$uncommented_stmt->execute();
$uncommented_calls = $uncommented_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$uncommented_stmt->close();

// Get commented calls
$commented_sql = "
    SELECT cl.id, cl.call_date, cl.client_contact, cl.duration, cl.status, cn.notes, cn.created_at as comment_date
    FROM call_logs cl 
    INNER JOIN call_notes cn ON cl.id = cn.call_id 
    WHERE cl.agent_contact = ? 
    ORDER BY cl.call_date DESC
";
$commented_stmt = $conn->prepare($commented_sql);
$commented_stmt->bind_param("s", $agent_phone);
$commented_stmt->execute();
$commented_calls = $commented_stmt->get_result()->fetch_all(MYSQLI_ASSOC);
$commented_stmt->close();

$uncommented_count = count($uncommented_calls);

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Calls - Agent Portal</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <div class="header">
        <div class="header-content">
            <h1>Call Feedback</h1>
            <a href="logout.php" class="logout-btn">Logout</a>
        </div>
    </div>

    <div class="container">
        <!-- Uncommented Calls Section -->
        <div class="section-header">
            <h2>Needs Feedback</h2>
            <span class="count-badge" id="uncommented-badge"><?php echo $uncommented_count; ?></span>
        </div>

        <div id="uncommented-section">
            <?php if (count($uncommented_calls) > 0): ?>
                <?php foreach ($uncommented_calls as $call): 
                    $duration_mins = floor($call['duration'] / 60);
                    $duration_secs = $call['duration'] % 60;
                ?>
                    <div class="call-card" data-call-id="<?php echo $call['id']; ?>">
                        <div class="call-card-header">
                            <div class="call-info">
                                <h3>
                                    <a href="tel:<?php echo htmlspecialchars($call['client_contact']); ?>" class="tel-link">
                                        <?php echo htmlspecialchars($call['client_contact']); ?>
                                    </a>
                                </h3>
                                <p><?php echo date('M d, Y - H:i', strtotime($call['call_date'])); ?></p>
                                <p>Duration: <?php echo sprintf("%02d:%02d", $duration_mins, $duration_secs); ?></p>
                            </div>
                            <button class="action-btn" onclick="openFeedbackModal(<?php echo $call['id']; ?>, '<?php echo htmlspecialchars($call['client_contact'], ENT_QUOTES); ?>', '<?php echo date('M d, Y H:i', strtotime($call['call_date'])); ?>', '<?php echo sprintf("%02d:%02d", $duration_mins, $duration_secs); ?>')">
                                Add Feedback
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-state-icon">✅</div>
                    <p>All calls have feedback!</p>
                </div>
            <?php endif; ?>
        </div>

        <!-- Commented Calls Section -->
        <div class="section-header" style="margin-top: 30px;">
            <h2>With Feedback</h2>
            <span class="count-badge success"><?php echo count($commented_calls); ?></span>
        </div>

        <div id="commented-section">
            <?php if (count($commented_calls) > 0): ?>
                <?php foreach ($commented_calls as $call): 
                    $duration_mins = floor($call['duration'] / 60);
                    $duration_secs = $call['duration'] % 60;
                ?>
                    <div class="call-card">
                        <div class="call-card-header">
                            <div class="call-info">
                                <h3>
                                    <a href="tel:<?php echo htmlspecialchars($call['client_contact']); ?>" class="tel-link">
                                        <?php echo htmlspecialchars($call['client_contact']); ?>
                                    </a>
                                </h3>
                                <p><?php echo date('M d, Y - H:i', strtotime($call['call_date'])); ?></p>
                                <p>Duration: <?php echo sprintf("%02d:%02d", $duration_mins, $duration_secs); ?></p>
                            </div>
                            <button class="action-btn view" onclick="viewFeedback(<?php echo $call['id']; ?>, '<?php echo htmlspecialchars($call['client_contact'], ENT_QUOTES); ?>', '<?php echo date('M d, Y H:i', strtotime($call['call_date'])); ?>', '<?php echo sprintf("%02d:%02d", $duration_mins, $duration_secs); ?>', '<?php echo htmlspecialchars($call['notes'], ENT_QUOTES); ?>', '<?php echo date('M d, Y H:i', strtotime($call['comment_date'])); ?>')">
                                View
                            </button>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="empty-state">
                    <div class="empty-state-icon">📝</div>
                    <p>No feedback submitted yet</p>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Feedback Modal -->
    <div id="feedbackModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Add Feedback</h3>
                <button class="close-modal" onclick="closeFeedbackModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="call-detail-row">
                    <label>Caller Number</label>
                    <div class="value" id="modal-caller"></div>
                </div>
                <div class="call-detail-row">
                    <label>Call Date</label>
                    <div class="value" id="modal-date"></div>
                </div>
                <div class="call-detail-row">
                    <label>Duration</label>
                    <div class="value" id="modal-duration"></div>
                </div>
                <div class="call-detail-row">
                    <label>Feedback / Notes</label>
                    <form id="feedbackForm" class="feedback-form">
                        <input type="hidden" id="call-id" name="call_id">
                        <textarea id="feedback-notes" name="notes" placeholder="Enter your feedback about this call..." required></textarea>
                    </form>
                </div>
            </div>
            <div class="modal-footer">
                <button class="btn-cancel" onclick="closeFeedbackModal()">Cancel</button>
                <button class="btn-submit" onclick="submitFeedback()">Submit Feedback</button>
            </div>
        </div>
    </div>

    <!-- View Modal -->
    <div id="viewModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Call Feedback</h3>
                <button class="close-modal" onclick="closeViewModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div class="call-detail-row">
                    <label>Caller Number</label>
                    <div class="value" id="view-caller"></div>
                </div>
                <div class="call-detail-row">
                    <label>Call Date</label>
                    <div class="value" id="view-date"></div>
                </div>
                <div class="call-detail-row">
                    <label>Duration</label>
                    <div class="value" id="view-duration"></div>
                </div>
                <div class="call-detail-row">
                    <label>Feedback</label>
                    <div class="value" id="view-notes" style="white-space: pre-wrap;"></div>
                </div>
                <div class="call-detail-row">
                    <label>Submitted On</label>
                    <div class="value" id="view-comment-date"></div>
                </div>
            </div>
            <div class="modal-footer">
                <button class="btn-cancel" onclick="closeViewModal()" style="width: 100%;">Close</button>
            </div>
        </div>
    </div>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <a href="home.php" class="nav-item">
            <div class="nav-icon">🏠</div>
            <div class="nav-label">Home</div>
        </a>
        <a href="calls.php" class="nav-item active">
            <div class="nav-icon">
                📋
                <?php if ($uncommented_count > 0): ?>
                    <span class="nav-badge" id="nav-badge"><?php echo $uncommented_count; ?></span>
                <?php endif; ?>
            </div>
            <div class="nav-label">Calls</div>
        </a>
        <a href="profile.php" class="nav-item">
            <div class="nav-icon">👤</div>
            <div class="nav-label">Profile</div>
        </a>
    </nav>

    <script>
        // Mobile viewport check
        function checkViewport() {
            if (window.innerWidth > 756) {
                document.body.innerHTML = `
                    <div class="error-404">
                        <h1>404</h1>
                        <p>This application is only accessible on mobile devices.</p>
                    </div>
                `;
            }
        }
        
        checkViewport();
        window.addEventListener('resize', checkViewport);

        // Open feedback modal
        function openFeedbackModal(callId, caller, date, duration) {
            document.getElementById('call-id').value = callId;
            document.getElementById('modal-caller').textContent = caller;
            document.getElementById('modal-date').textContent = date;
            document.getElementById('modal-duration').textContent = duration;
            document.getElementById('feedback-notes').value = '';
            document.getElementById('feedbackModal').classList.add('active');
        }

        // Close feedback modal
        function closeFeedbackModal() {
            document.getElementById('feedbackModal').classList.remove('active');
        }

        // View feedback modal
        function viewFeedback(callId, caller, date, duration, notes, commentDate) {
            document.getElementById('view-caller').textContent = caller;
            document.getElementById('view-date').textContent = date;
            document.getElementById('view-duration').textContent = duration;
            document.getElementById('view-notes').textContent = notes;
            document.getElementById('view-comment-date').textContent = commentDate;
            document.getElementById('viewModal').classList.add('active');
        }

        // Close view modal
        function closeViewModal() {
            document.getElementById('viewModal').classList.remove('active');
        }

        // Submit feedback
        function submitFeedback() {
            const form = document.getElementById('feedbackForm');
            const callId = document.getElementById('call-id').value;
            const notes = document.getElementById('feedback-notes').value.trim();

            if (!notes) {
                alert('Please enter feedback before submitting.');
                return;
            }

            // Disable submit button
            event.target.disabled = true;
            event.target.textContent = 'Submitting...';

            // Send AJAX request
            fetch('submit_feedback.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded',
                },
                body: `call_id=${callId}&notes=${encodeURIComponent(notes)}`
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    closeFeedbackModal();
                    // Reload page to update counts and move card
                    location.reload();
                } else {
                    alert('Error: ' + data.message);
                    event.target.disabled = false;
                    event.target.textContent = 'Submit Feedback';
                }
            })
            .catch(error => {
                console.error('Error:', error);
                alert('An error occurred. Please try again.');
                event.target.disabled = false;
                event.target.textContent = 'Submit Feedback';
            });
        }

        // Auto-refresh for new calls (every 30 seconds)
        setInterval(() => {
            if (!document.hidden) {
                location.reload();
            }
        }, 30000);

        // Close modals on outside click
        document.getElementById('feedbackModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeFeedbackModal();
            }
        });

        document.getElementById('viewModal').addEventListener('click', function(e) {
            if (e.target === this) {
                closeViewModal();
            }
        });
    </script>
</body>
</html>